/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 *                                                                          *
 * FILE NAME:   IAI_EZMac_PIC.h                                             *
 *                                                                          *
 * DESIGNER:	Kovacs, Krisztian; Lukacs, Miklos                           *
 * PROGRAMMER: Kovacs Krisztian;                                            *
 *                                                                          *
 * DESCRIPTION: EZMac PIC processor dependend functions                     *
 *                                                                          *
 * REVISION:    01_7  Nov  22, 2005     created by krk                      *
 *				01_7  Dec  08, 2005	Port to HiTech						    *
 *                                                                          *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/***************************************************************************\
                             Find out the compiler
\***************************************************************************/
#ifdef HI_TECH_C
 #define HITECH_COMPILER
 #if defined (_PIC18)
  #define HITECH_PIC18
 #else
  #define HITECH_PIC16
 #endif
#else
 #if defined(__PCB__) || defined(__PCM__) || defined(__PCH__) || defined(__PCW__) || defined(__PCWH__)
  #define CCS_C_COMPILER
 #endif
#endif
 

#ifdef HITECH_PIC18
 #define BANK_1
 #define BANK_2
 #define _NEAR__ near
#endif
#ifdef HITECH_PIC16
 #define BANK_1 bank1
 #define BANK_2 bank2
 #define _NEAR__
#endif

/*If the memory addresses are allocated manualy this definition has to be uncommented*/
#ifdef CCS_C_COMPILER
//#define FIX_MEM_ADDRESS
#endif

/*oscillator frequency*/
#define CLOCK_FREQ      4000000

//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//                          Set the hardware
//Uncomment this, and set up SPI I/O definitions in EZMac_Internal.h and in IAI_EZMac_PIC.h!
//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#define END_HARDWARE


/***************************************************************************\
                              Program start up
\***************************************************************************/
/*set processor type*/
#ifdef CCS_C_COMPILER
 #ifdef END_HARDWARE
  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  //                This is a sample config setting, it should be
  //              changed according to the application Hardware!!!
  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  #include <16F916.h>
  #pragma device  *=16
 #else
  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  //              This is the debug config setting, can be
  //                       omitted if confusing!!!
  //!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  #include <16F876.h>
  #pragma device  *=16
 #endif
#endif


#include "IAI_EZMac_PIC.h"

/**************************************************************************\
						        Set fuses    			  
\**************************************************************************/
#ifdef CCS_C_COMPILER
 #ifdef END_HARDWARE
  #pragma fuses INTRC_IO,nowdt,put,noprotect,nocpd,nobrownout,noieso,NOMCLR
 #else
  #pragma fuses hs,nowdt,noprotect,nobrownout,put,nolvp,nowrt,nocpd
 #endif
#endif

#ifdef HITECH_COMPILER
 #ifdef END_HARDWARE
   //__CONFIG(OSCSDIS&HS&WDTDIS&PWRTDIS&DUNPROT&UNPROTECT&BORDIS&MCLREN);
 #else
   __CONFIG(HS&WDTDIS&DUNPROT&BORDIS&PWRTEN&LVPDIS&WRTDIS&UNPROTECT);
 #endif
#endif

/**************************************************************************\
   			        HiTech compiler dependent functions  			  
\**************************************************************************/
#ifdef HITECH_PIC18
/*handle Timer1*/

 void set_timer1(int16 value)
  {
   TMR1H=(value>>8)&0x0FF; TMR1L=value&0x0FF;
  }
  #pragma inline(set_timer1)

/*handle the ADC*/
  int16 read_adc(int8 mode)
  {
   switch(mode)
   {
    case ADC_START_ONLY:
    {
     ADON=1;
     GODONE=1;
     return 0;
    }
    case ADC_START_AND_READ:
    {
     ADON=1;
     GODONE=1;
     while(GODONE) continue;
     return (((int16)ADRESH)<<8)|((int16)ADRESL);
    }
    case ADC_READ_ONLY:
    {
     while(GODONE) continue;
     return (((int16)ADRESH)<<8)|((int16)ADRESL);
    }
   }
   return 0;
  }

 #ifdef EZMAC_BOARD_WITH_ADC 
  #pragma interrupt_level 1
 #endif
  void setup_adc(int8 mode)
  {
   ADCON2=0xa0;
   ADCON2|=mode;
  }
  #pragma inline(setup_adc)
 
  void setup_adc_ports(int8 ports)
  {
   ADCON1=0x00|ports;
  }
  #pragma inline(setup_adc_ports)
 
  void set_adc_channel(int8 ch)
  {
   ADCON0&=0xC3;
   ADCON0|=(ch<<2);
  }
  #pragma inline(setup_adc_channel)

#endif

#ifdef HITECH_PIC16
/*Handle Timer1*/
void set_timer1(int16 value)
  {
   TMR1H=(value>>8)&0x0FF; TMR1L=value&0x0FF;
  }
  #pragma inline(set_timer1)

/*handle the ADC*/
 int16 read_adc(int8 mode)
 {
  switch(mode)
  {
   case ADC_START_ONLY:
   {
    ADON=1;
    ADGO=1;
    return 0;
   }
   case ADC_START_AND_READ:
   {
    ADON=1;
    ADGO=1;
    while(ADGO) continue;
    return (((int16)ADRESH)<<8)|((int16)ADRESL);
   }
   case ADC_READ_ONLY:
   {
    while(ADGO) continue;
    return (((int16)ADRESH)<<8)|((int16)ADRESL);
   }
  }
  return 0;
 }

 #ifdef EZMAC_BOARD_WITH_ADC 
  #pragma interrupt_level 1
 #endif
 void setup_adc(int8 mode)
 {
  ADCON0&=0x3F;
  ADCON0|=mode;
 }
 #pragma inline(setup_adc)

 void setup_adc_ports(int8 ports)
 {
  ADCON1=0x80|ports;
 }
 #pragma inline(setup_adc_ports)

 void set_adc_channel(int8 ch)
 {
  ADCON0&=0xC0;
  ADCON0|=ch;
 }
 #pragma inline(setup_adc_channel)
#endif

#ifdef HITECH_COMPILER
 #if CLOCK_FREQ >= 12000000
  #define	delay_us(x)	{ unsigned char _dcnt; \
			  _dcnt = (x)*((CLOCK_FREQ)/(12000000)); \
			  while(--_dcnt != 0) \
				  continue; }
 #else

  #define	delay_us(x)	{ unsigned char _dcnt; \
			  _dcnt = (x)/((12000000)/(CLOCK_FREQ))|1; \
			  while(--_dcnt != 0) \
				  continue; }
 #endif

 extern void delay_ms(unsigned char cnt);

 void delay_ms(unsigned char cnt)
 {
 #if CLOCK_FREQ <= 2000000
  do {
   delay_us(996);
  } while(--cnt);
 #endif

 #if CLOCK_FREQ > 2000000
  unsigned char i;
  do {
   i = 4;
   do {
    delay_us(250);
   } while(--i);
  } while(--cnt);
 #endif
 }
#pragma inline(delay_ms)

 /*handle uart*/
 void init_uart(void)
 {
  TRISC6=0;
  TRISC7=1;
  BRGH=1;
  SPBRG=DIVIDE;
  TXEN=1;
  SPEN=1;
  CREN=1;
 }

 void putch(unsigned char ch) 
 {
  /* output one byte */
  while(!TRMT) /* set when register is empty */
   continue;
  TXREG = ch;

  TRISC5 = 0;
  RC5 = !RC5;
 }
//#pragma inline(putch)

// getch added by Balazs Scherer on 17.01.06
 static char tx_len=0;
 static char *tx_data=0;

 void TX_int(void)
 {
   if(tx_len>0)
   {
     TXREG=*tx_data++;
     tx_len--;
   }
   else
   {
     disable_INT_TBE;
     tx_len=0;
   }
 }

 static char buffer[32];
 static char read=0;
 static char write=0;

 void RC_FIFO_int(void)
 {
   char ch;
   char err=0;

   if(OERR==1)
   {
     CREN=0;
     CREN=1;
     err=1;
   }    
   if(FERR==1)
   {
     ch=RCREG;
     err=1;
   }

   if(err==0)
   {
     ch=RCREG;
     buffer[write]=ch;
     write++;
     write&=0x1F;
   }
 }

 unsigned char getch(void) 
 {
  char ch;
  /* wait for one byte */
  while(read==write) 
   continue;

  ch=buffer[read];
  read++;
  read&=0x1F;

  return ch;
 }

// timer0_start(unsigned int timeout_ms) added by Balazs Scherer on 13.02.06
// maximum timeout is about 8s at 4 MHz in this configuration
void timer0_start(unsigned int timeout_ms)
 {
	unsigned int t_ms;
	unsigned char tl,th;
	disable_INT_TIMER0;

	// configurating timer0 
	T0CON = 0x05; //

	// Setting the timeout
	t_ms = (unsigned int)(CLOCK_FREQ >> 18) * timeout_ms;
	th = (t_ms >> 8);
	tl = (unsigned char)(t_ms & 0xFF);

	TMR0H = ~th;
	TMR0L = ~tl;

	// Starting the timer
	TMR0ON = 1;
	

 }

// timer0_stop() added by Balazs Scherer on 13.02.06
void timer0_stop(void)
 {
	TMR0ON = 0;
	TMR0IF = 0;
 }


// getch_timeout added by Balazs Scherer on 13.02.06
 unsigned char getch_timeout(unsigned char* ch) 
 {
 unsigned char error;

	timer0_start(100);

	  /* wait for one byte or timeout*/
	  while( !RCIF && !TMR0IF);

	  if (RCIF)
		{
		  TRISC2 = 0;
	 	  RC2 = !RC2;
		  error = 0;
		}
		else
		{
		  error = 1;
		  CREN=0;
		  CREN=1;
		}

	*ch = RCREG;

	timer0_stop();
	return error;

 }



#endif

/**************************************************************************\
   			        Set peripherals  			  
\**************************************************************************/
#ifdef CCS_C_COMPILER
 #pragma use delay(clock=CLOCK_FREQ)        //fuses + delay sets the OSCCON also
 #pragma use rs232(baud=19200,xmit=PIN_C6,rcv=PIN_C7)
 #ifdef FIX_MEM_ADDRESS
  #pragma reserve 0x29:0x75
 #endif
#endif

/**************************************************************************\
   			        Set port pins  			  
\**************************************************************************/
#ifdef CCS_C_COMPILER
 #pragma byte rcsta = 0x18
 #pragma byte rcreg = 0x1A
 #pragma bit oerr = rcsta.1
 #pragma bit ferr = rcsta.2
 #pragma bit cren = rcsta.4

 #pragma use standard_io(A)
 #pragma use standard_io(B)
 #ifdef END_HARDWARE
  #pragma use standard_io(C)
 #endif
#endif

#ifdef HITECH_COMPILER
 static volatile unsigned char rcsta @ 0x18;
 static volatile unsigned char rcreg @ 0x1A;
 static volatile bit oerr @ (unsigned)&rcsta*8+1;
 static volatile bit ferr @ (unsigned)&rcsta*8+2;
 static volatile bit cren @ (unsigned)&rcsta*8+4;
#endif

/**************************************************************************\
   		     			        Include files   			  
\**************************************************************************/
#ifdef CCS_C_COMPILER
// #include "IAI_EZMac_PIC.h"
 #include "IAI_EZMac_Lite.h"
 #include "IAI_EZMac_Lite_Int.h"
 #include <string.h>
 #include "IAI_EZMac_Lite_Int.c"
 #include "IAI_EZMac_Lite.c"
#endif

#ifdef HITECH_COMPILER
// #include "IAI_EZMac_PIC.h"
 #include "IAI_EZMac_Lite.h"
 #include "IAI_EZMac_Lite_Int.h"
 #include <string.h>
 #include <stdio.h>
 #include "IAI_EZMac_Lite_Int.c"
 #include "IAI_EZMac_Lite.c"
#endif


/**************************************************************************\
   			        Declare interrupt functions			  
\**************************************************************************/
#ifdef CCS_C_COMPILER
 #pragma priority rtcc, rda
#endif
#ifdef HITECH_COMPILER
#endif

#ifdef CCS_C_COMPILER
 // External interrupt routine
 #pragma int_ext
 void Ext_ISR (void)
 {
    EZMac_CommISR ();
 }
 // peripheral interrupt routines
 #pragma int_timer1
 void TimerISR (void)
 {
    EZMac_TimerISR ();
 }
#endif

#ifdef HITECH_COMPILER
 void Ext_ISR (void)
 {
    EZMac_CommISR ();
 }
// #pragma inline(Ext_ISR)

 void TimerISR (void)
 {
    EZMac_TimerISR ();
 }
// #pragma inline(TimerISR)

 #define INT_EXT_ISR Ext_ISR
 #define INT_TIMER1_ISR TimerISR

/*
 #ifdef END_HARDWARE
  void MAIN_hardint(void);
  #define INT_TIMER0_ISR MAIN_hardint
 #else
  void MAIN_hardint(void);
  #define INT_TIMER2_ISR MAIN_hardint
 #endif                                //#ifdef END_HARDWARE
*/

 #define INT_RDA_ISR RC_FIFO_int
 #define INT_TBE_ISR TX_int

 #pragma interrupt_level 1
 void interrupt ISR(void)
 {
 #ifdef INT_RDA_ISR
  if(RCIE==1 && RCIF==1)
  {
    RCIF=0;
    INT_RDA_ISR();
  }
 #else
  if(RCIE==1 && RCIF==1)
  {
    RCIF=0;
  }
 #endif
 #ifdef INT_TIMER0_ISR
  if(TMR0IE==1 && TMR0IF==1)
  {
   TMR0IF=0;
   INT_TIMER0_ISR();
  }
 #else
  if(TMR0IE==1 && TMR0IF==1)
  {
   TMR0IF=0;
  }
 #endif
 #ifdef INT_RTCC_ISR
  if(TMR0IE==1 && TMR0IF==1)
  {
   TMR0IF=0;
   INT_RTCC_ISR();
  }
 #else
  if(TMR0IE==1 && TMR0IF==1)
  {
   TMR0IF=0;
  }
 #endif
#ifdef END_HARDWARE
 #ifdef INT_EXT_ISR
  if(INT0IE==1 && INT0IF==1)
  {
    INT0IF=0;
    INT_EXT_ISR();
  }
 #else
  if(INT0IE==1 && INT0IF==1)
  {
    INT0IF=0;
  }
 #endif
#else
 #ifdef INT_EXT_ISR
  if(INTE==1 && INTF==1)
  {
    INTF=0;
    INT_EXT_ISR();
  }
 #else
  if(INTE==1 && INTF==1)
  {
    INTF=0;
  }
 #endif
#endif
 #ifdef INT_RB_ISR
  if(RBIE==1 && RBIF==1)
  {
    RBIF=0;
    INT_RB_ISR();
  }
 #else
  if(RBIE==1 && RBIF==1)
  {
    RBIF=0;
  }
 #endif
 #ifdef __PINS_40
  #ifdef INT_PSP_ISR
   if(PSPIE==1 && PSPIF==1)
   {
     PSPIF=0;
     INT_PSP_ISR();
   }
  #else
   if(PSPIE==1 && PSPIF==1)
   {
     PSPIF=0;
   }
  #endif
 #endif
 #ifdef INT_AD_ISR
  if(ADIE==1 && ADIF==1)
  {
    ADIF=0;
    INT_AD_ISR();
  }
 #else
  if(ADIE==1 && ADIF==1)
  {
    ADIF=0;
  }
 #endif
 #ifdef INT_SSP_ISR
  if(SSPIE==1 && SSPIF==1)
  {
    SSPIF=0;
    INT_SSP_ISR();
  }
 #else
  if(SSPIE==1 && SSPIF==1)
  {
    SSPIF=0;
  }
 #endif
 #ifdef INT_CCP1_ISR
  if(CCP1IE==1 && CCP1IF==1)
  {
    CCP1IF=0;
    INT_CCP1_ISR();
  }
 #else
  if(CCP1IE==1 && CCP1IF==1)
  {
    CCP1IF=0;
  }
 #endif
 #ifdef INT_TIMER2_ISR
  if(TMR2IE==1 && TMR2IF==1)
  {
   TMR2IF=0;
   INT_TIMER2_ISR();
  }
 #else
  if(TMR2IE==1 && TMR2IF==1)
  {
   TMR2IF=0;
  }
 #endif
 #ifdef INT_TIMER1_ISR
  if(TMR1IE==1 && TMR1IF==1)
  {
   TMR1IF=0;
   INT_TIMER1_ISR();
  }
 #else
  if(TMR1IE==1 && TMR1IF==1)
  {
   TMR1IF=0;
  }
 #endif
 #ifdef INT_EEPROM_ISR
  if(EEIE==1 && EEIF==1)
  {
   EEIF=0;
   INT_EEPROM_ISR();
  }
 #else
  if(EEIE==1 && EEIF==1)
  {
   EEIF=0;
  }
 #endif
 #ifdef INT_BUSCOL_ISR
  if(BCLIE==1 && BCLIF==1)
  {
   BCLIF=0;
   INT_BLC_ISR();
  }
 #else
  if(BCLIE==1 && BCLIF==1)
  {
   BCLIF=0;
  }
 #endif
 #ifdef INT_CCP2_ISR
  if(CCP2IE==1 && CCP2IF==1)
  {
   CCP2IF=0;
   INT_CCP2_ISR();
  }
 #else
  if(CCP2IE==1 && CCP2IF==1)
  {
   CCP2IF=0;
  }
 #endif
 #ifdef INT_TBE_ISR
  if(TXIE==1 && TXIF==1)
  {
    TXIF=0;
    INT_TBE_ISR();
  }
 #else
  if(TXIE==1 && TXIF==1)
  {
    TXIF=0;
  }
 #endif
 }
#endif

/**************************************************************************\
   			          SPI port handling
\**************************************************************************/
#ifdef SW_SPI
 /*software SPI is used (bit bang mode)*/
 //Changed by Laszlo Kadar on 29.11.05
#ifdef HITECH_COMPILER
 static _NEAR__ reg_union BANK_2 u_reg;
#endif
#ifdef CCS_C_COMPILER
 static reg_union u_reg;
#endif
 #ifdef FIX_MEM_ADDRESS
  #ifdef CCS_C_COMPILER
   #pragma locate u_reg=0x62
  #endif
 #endif
 int8 Spi_Comm0(void)
 {
   MOSI_PIN = 0;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit7 = MISO_PIN;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit6 = MISO_PIN;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit5 = MISO_PIN;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit4 = MISO_PIN;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit3 = MISO_PIN;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit2 = MISO_PIN;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit1 = MISO_PIN;
   CLK_PIN = 0;

   CLK_PIN = 1;
   u_reg.testreg.bit0 = MISO_PIN;
   CLK_PIN = 0;

  return u_reg.adat;
 }

 //===============================================================================================

// Changed by Laszlo Kadar on 26.11.05
#ifdef HITECH_COMPILER
 #pragma interrupt_level 1
#endif
 void EZ_Spi_Write(int8 data)
 {
// Changed by Laszlo Kadar on 26.11.05
   u_reg.adat = data;

   MOSI_PIN = 0;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit7 ;
   CLK_PIN = 1;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit6 ;
   CLK_PIN = 1;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit5 ;
   CLK_PIN = 1;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit4 ;
   CLK_PIN = 1;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit3 ;
   CLK_PIN = 1;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit2 ;
   CLK_PIN = 1;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit1 ;
   CLK_PIN = 1;
   CLK_PIN = 0;

   MOSI_PIN = u_reg.testreg.bit0 ;
   CLK_PIN = 1;
   CLK_PIN = 0;
 }
#else
 /*hardware SPI is used*/
 //HW SPI port doesn't supported now!
 int8 Spi_Comm0(void)
 {
 //activate HW SPI port

 //read the data

 //deactivate HW SPI port
 }

 //===============================================================================================

 void EZ_Spi_Write(int data)
 {
 //activate HW SPI port

 //write the data

 //deactivate HW SPI port
 }
#endif
